document.addEventListener('DOMContentLoaded', function() {
    const apiKeyInput = document.getElementById('api-key');
    const saveApiKeyBtn = document.getElementById('save-api-key');
    const testConnectionBtn = document.getElementById('test-connection');
    const apiMessage = document.getElementById('api-message');
    const statusContainer = document.getElementById('status-container');
    const profileSection = document.getElementById('profile-section');
    const linkedinProfileInfo = document.getElementById('linkedin-profile-info');
    const listSelect = document.getElementById('list-select');
    const addToListBtn = document.getElementById('add-to-list');
    const moveToListBtn = document.getElementById('move-to-list');
    const addMessage = document.getElementById('add-message');
    const existingLeadInfo = document.getElementById('existing-lead-info');
    const apiConfigHeader = document.getElementById('api-config-header');
    const apiConfigContent = document.getElementById('api-config-content');
    const apiToggle = document.getElementById('api-toggle');

    let currentProfile = null;
    let lists = [];
    let existingLead = null;

    // Load saved API key and check if config should be collapsed
    chrome.storage.sync.get(['grinfiApiKey'], function(result) {
        if (result.grinfiApiKey) {
            apiKeyInput.value = result.grinfiApiKey;
            
            // Check if we should keep config collapsed
            chrome.storage.local.get(['apiConfigCollapsed'], function(localResult) {
                if (localResult.apiConfigCollapsed) {
                    apiConfigContent.classList.add('collapsed');
                    apiToggle.classList.add('collapsed');
                }
            });
            
            testConnection();
        }
    });

    // Add collapsible functionality
    apiConfigHeader.addEventListener('click', function() {
        const isCollapsed = apiConfigContent.classList.contains('collapsed');
        
        if (isCollapsed) {
            apiConfigContent.classList.remove('collapsed');
            apiToggle.classList.remove('collapsed');
        } else {
            apiConfigContent.classList.add('collapsed');
            apiToggle.classList.add('collapsed');
        }
    });

    // Check if we're on a LinkedIn profile page
    chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
        console.log('Grinfi Extension Popup: Current tab URL:', tabs[0].url);
        
        if (tabs[0].url.includes('linkedin.com/in/')) {
            console.log('Grinfi Extension Popup: On LinkedIn profile page');
            
            // First, try to inject content script if it's not already there
            chrome.scripting.executeScript({
                target: {tabId: tabs[0].id},
                files: ['content.js']
            }).then(() => {
                console.log('Content script injected/reloaded');
                
                // Wait a bit for the script to initialize
                setTimeout(() => {
                    // Get profile data from content script
                    chrome.tabs.sendMessage(tabs[0].id, {action: 'getProfileData'}, function(response) {
                        console.log('Grinfi Extension Popup: Response from content script:', response);
                        
                        if (chrome.runtime.lastError) {
                            console.error('Grinfi Extension Popup: Error communicating with content script:', chrome.runtime.lastError);
                            // Show profile section anyway for debugging
                            showFallbackProfileSection();
                            return;
                        }
                        
                        if (response && response.profile) {
                            currentProfile = response.profile;
                            console.log('Grinfi Extension Popup: Profile data received:', currentProfile);
                            displayLinkedInProfile(currentProfile);
                            profileSection.classList.remove('hidden');
                            loadLists().then(() => {
                                checkExistingLead(currentProfile);
                            });
                        } else {
                            console.log('Grinfi Extension Popup: No profile data received');
                            if (response && response.error) {
                                console.error('Grinfi Extension Popup: Error from content script:', response.error);
                            }
                            // Show fallback profile section
                            showFallbackProfileSection();
                        }
                    });
                }, 500);
            }).catch(error => {
                console.error('Failed to inject content script:', error);
                // Show fallback profile section
                showFallbackProfileSection();
            });
        } else {
            console.log('Grinfi Extension Popup: Not on LinkedIn profile page');
        }
    });

    saveApiKeyBtn.addEventListener('click', function() {
        const apiKey = apiKeyInput.value.trim();
        if (apiKey) {
            chrome.storage.sync.set({grinfiApiKey: apiKey}, function() {
                showMessage(apiMessage, 'API key saved successfully', 'success');
                testConnection();
            });
        } else {
            showMessage(apiMessage, 'Please enter an API key', 'error');
        }
    });

    testConnectionBtn.addEventListener('click', testConnection);

    addToListBtn.addEventListener('click', function() {
        const selectedListId = listSelect.value;
        if (!selectedListId) {
            showMessage(addMessage, 'Please select a list', 'error');
            return;
        }

        if (!currentProfile) {
            showMessage(addMessage, 'No LinkedIn profile data found', 'error');
            return;
        }

        addToListBtn.disabled = true;
        addToListBtn.textContent = 'Adding...';

        addLeadToList(currentProfile, selectedListId);
    });

    moveToListBtn.addEventListener('click', function() {
        const selectedListId = listSelect.value;
        if (!selectedListId) {
            showMessage(addMessage, 'Please select a list', 'error');
            return;
        }

        if (!currentProfile || !existingLead) {
            showMessage(addMessage, 'No profile or lead data found', 'error');
            return;
        }

        moveToListBtn.disabled = true;
        moveToListBtn.textContent = 'Moving...';

        moveLeadToList(currentProfile, selectedListId);
    });

    function testConnection() {
        const apiKey = apiKeyInput.value.trim();
        if (!apiKey) {
            showMessage(apiMessage, 'Please enter an API key first', 'error');
            return;
        }

        testConnectionBtn.disabled = true;
        testConnectionBtn.textContent = 'Testing...';

        // Test connection with a simple lead search
        fetch('https://leadgen.grinfi.io/leads/api/leads/search', {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${apiKey}`,
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                limit: 1,
                offset: 0
            })
        })
        .then(async response => {
            if (response.ok) {
                showStatus('Connected to Grinfi API', 'connected');
                showMessage(apiMessage, 'Connection successful!', 'success');
                loadLists();
                
                // Auto-collapse API config after successful connection and keep it collapsed
                setTimeout(() => {
                    apiConfigContent.classList.add('collapsed');
                    apiToggle.classList.add('collapsed');
                    
                    // Store that we should keep it collapsed
                    chrome.storage.local.set({apiConfigCollapsed: true});
                }, 1500);
                
                return response.json();
            } else {
                const errorText = await response.text();
                console.error('API Error:', response.status, errorText);
                throw new Error(`HTTP ${response.status}: ${errorText.substring(0, 100)}`);
            }
        })
        .catch(error => {
            showStatus('Not connected to Grinfi API', 'disconnected');
            console.error('Connection error:', error);
            showMessage(apiMessage, `Connection failed: ${error.message}`, 'error');
        })
        .finally(() => {
            testConnectionBtn.disabled = false;
            testConnectionBtn.textContent = 'Test Connection';
        });
    }

    function loadLists() {
        return new Promise(async (resolve, reject) => {
            chrome.storage.sync.get(['grinfiApiKey'], async function(result) {
                if (!result.grinfiApiKey) {
                    reject('No API key found');
                    return;
                }

                try {
                    // Загружаем ВСЕ списки с пагинацией
                    let allLists = [];
                    let offset = 0;
                    const limit = 20; // Как показано в документации по умолчанию 20
                    let hasMore = true;

                    console.log('Loading all lists with pagination...');

                    while (hasMore) {
                        const response = await fetch(`https://leadgen.grinfi.io/leads/api/lists?limit=${limit}&offset=${offset}`, {
                            method: 'GET',
                            headers: {
                                'Authorization': `Bearer ${result.grinfiApiKey}`,
                                'Content-Type': 'application/json'
                            }
                        });

                        if (!response.ok) {
                            const errorText = await response.text();
                            console.error('Lists API Error:', response.status, errorText);
                            throw new Error(`Failed to load lists: ${response.status}`);
                        }

                        const data = await response.json();
                        const currentBatch = Array.isArray(data) ? data : (data.data || []);
                        
                        console.log(`Loaded batch: ${currentBatch.length} lists`);
                        allLists = allLists.concat(currentBatch);
                        
                        // Если получили меньше чем limit, значит это последняя страница
                        hasMore = currentBatch.length === limit;
                        offset += limit;
                        
                        // Защита от бесконечного цикла
                        if (offset > 1000) {
                            console.log('Reached max offset, stopping');
                            break;
                        }
                    }
                    
                    console.log(`Total lists loaded: ${allLists.length}`);
                    lists = allLists;
                    populateListSelect();
                    resolve(lists);
                } catch (error) {
                    console.error('Error loading lists:', error);
                    listSelect.innerHTML = '<option value="">Error loading lists</option>';
                    reject(error);
                }
            });
        });
    }

    function populateListSelect() {
        listSelect.innerHTML = '<option value="">Select a list...</option>';
        
        console.log('Populating list select with lists:', lists);
        
        if (lists && lists.length > 0) {
            lists.forEach((list, index) => {
                console.log(`List ${index}:`, list);
                const option = document.createElement('option');
                option.value = list.uuid || list.id;
                option.textContent = list.name || list.title || `List ${list.uuid || list.id}`;
                listSelect.appendChild(option);
            });
            addToListBtn.disabled = false;
        } else {
            listSelect.innerHTML = '<option value="">No lists found</option>';
            console.log('No lists available');
        }
    }

    function displayLinkedInProfile(profile) {
        // Check if edit button should be shown - only for NEW leads (when existingLead is null)
        const showEditButton = !existingLead;
        
        // Final refined check for empty or placeholder values
        const isInvalid = (val) => !val || val === 'Unknown Name' || val === 'Not specified' || val.trim() === '';
        
        const nameDisplay = !isInvalid(profile.name) ? profile.name : '<span style="color: #888; font-style: italic;">Will be parsed by Grinfi</span>';
        const positionDisplay = !isInvalid(profile.position) ? profile.position : '<span style="color: #888; font-style: italic;">Will be parsed by Grinfi</span>';
        const companyDisplay = !isInvalid(profile.company) ? profile.company : '<span style="color: #888; font-style: italic;">Will be parsed by Grinfi</span>';

        linkedinProfileInfo.innerHTML = `
            <div style="display: flex; justify-content: space-between; align-items: flex-start;">
                <div style="flex: 1;">
                    <h4 id="profile-name">${nameDisplay}</h4>
                    <p><strong>Position:</strong> <span id="profile-position">${positionDisplay}</span></p>
                    <p><strong>Company:</strong> <span id="profile-company">${companyDisplay}</span></p>
                    <p><strong>LinkedIn URL:</strong> ${profile.linkedinUrl || 'Current page'}</p>
                </div>
                ${showEditButton ? `<button id="edit-profile-btn" style="padding: 6px 8px; font-size: 12px; cursor: pointer; border: 1px solid #ddd; background: #f8f9fa; border-radius: 3px;" title="Edit profile data">✏️</button>` : ''}
            </div>
            <div class="automation-info" style="margin-top: 10px; border-left: 3px solid #0066cc;">
                Any missing data will be automatically enriched by Grinfi platform after adding.
            </div>
            ${showEditButton ? `<div id="edit-profile-form" class="hidden" style="margin-top: 15px; padding: 10px; background: #f8f9fa; border-radius: 6px; border: 1px solid #ddd;">
                <h5 style="margin: 0 0 10px 0;">Edit Profile Data</h5>
                <div class="input-group">
                    <label for="edit-first-name">First Name:</label>
                    <input type="text" id="edit-first-name" value="${profile.firstName || profile.name?.split(' ')[0] || ''}" style="width: 100%; padding: 6px; border: 1px solid #ddd; border-radius: 3px;">
                </div>
                <div class="input-group">
                    <label for="edit-last-name">Last Name:</label>
                    <input type="text" id="edit-last-name" value="${profile.lastName || profile.name?.split(' ').slice(1).join(' ') || ''}" style="width: 100%; padding: 6px; border: 1px solid #ddd; border-radius: 3px;">
                </div>
                <div class="input-group">
                    <label for="edit-position">Position:</label>
                    <input type="text" id="edit-position" value="${profile.position || ''}" style="width: 100%; padding: 6px; border: 1px solid #ddd; border-radius: 3px;">
                </div>
                <div class="input-group">
                    <label for="edit-company">Company:</label>
                    <input type="text" id="edit-company" value="${profile.company || ''}" style="width: 100%; padding: 6px; border: 1px solid #ddd; border-radius: 3px;">
                </div>
                <div style="margin-top: 10px;">
                    <button id="save-profile-btn" style="background-color: #28a745; color: white; padding: 6px 12px; border: none; border-radius: 3px; cursor: pointer; margin-right: 5px;">Save</button>
                    <button id="cancel-edit-btn" style="background-color: #6c757d; color: white; padding: 6px 12px; border: none; border-radius: 3px; cursor: pointer;">Cancel</button>
                </div>
            </div>` : ''}
        `;
        
        // Add event listeners for edit functionality (only if edit button exists)
        const editBtn = document.getElementById('edit-profile-btn');
        if (editBtn) {
            editBtn.addEventListener('click', function() {
                document.getElementById('edit-profile-form').classList.remove('hidden');
                this.style.display = 'none';
            });
            
            document.getElementById('cancel-edit-btn').addEventListener('click', function() {
                document.getElementById('edit-profile-form').classList.add('hidden');
                document.getElementById('edit-profile-btn').style.display = 'block';
            });
            
            document.getElementById('save-profile-btn').addEventListener('click', function() {
                // Get edited values
                const editedFirstName = document.getElementById('edit-first-name').value;
                const editedLastName = document.getElementById('edit-last-name').value;
                const editedPosition = document.getElementById('edit-position').value;
                const editedCompany = document.getElementById('edit-company').value;
                
                // Combine first and last name for full name
                const fullName = [editedFirstName, editedLastName].filter(n => n.trim()).join(' ');
                
                const editedProfile = {
                    ...profile,
                    name: fullName,
                    firstName: editedFirstName,
                    lastName: editedLastName,
                    position: editedPosition,
                    company: editedCompany,
                    headline: editedPosition // Update headline to match position
                };
                
                // Update display
                document.getElementById('profile-name').innerHTML = (editedProfile.name && editedProfile.name !== 'Unknown Name' && editedProfile.name.trim() !== '') ? editedProfile.name : '<span style="color: #888; font-style: italic;">Will be parsed by Grinfi</span>';
                document.getElementById('profile-position').innerHTML = (editedProfile.position && editedProfile.position !== 'Not specified' && editedProfile.position.trim() !== '') ? editedProfile.position : '<span style="color: #888; font-style: italic;">Will be parsed by Grinfi</span>';
                document.getElementById('profile-company').innerHTML = (editedProfile.company && editedProfile.company !== 'Not specified' && editedProfile.company.trim() !== '') ? editedProfile.company : '<span style="color: #888; font-style: italic;">Will be parsed by Grinfi</span>';
                
                // Update the global currentProfile variable
                currentProfile = editedProfile;
                
                // Hide form
                document.getElementById('edit-profile-form').classList.add('hidden');
                document.getElementById('edit-profile-btn').style.display = 'block';
                
                console.log('Profile updated with separate first/last names:', editedProfile);
            });
        }
    }

    function addLeadToList(profile, listUuid) {
        chrome.storage.sync.get(['grinfiApiKey'], function(result) {
            if (!result.grinfiApiKey) {
                showMessage(addMessage, 'API key not found', 'error');
                resetAddButton();
                return;
            }

            // Use currentProfile if available (after editing), otherwise use passed profile
            const profileToUse = currentProfile || profile;
            
            console.log('=== ADDING LEAD TO LIST ===');
            console.log('Original profile:', profile);
            console.log('Current profile (after editing):', currentProfile);
            console.log('Profile to use:', profileToUse);
            
            // Ensure all fields are clean and don't contain placeholders
            const cleanValue = (val) => {
                const s = (val || '').trim();
                if (s === 'Not specified' || s === 'Unknown Name' || s.includes('Will be parsed by Grinfi')) return '';
                return s;
            };

            const leadData = {
                lead: {
                    linkedin_id: profileToUse.linkedinUrl || window.location.href,
                    first_name: cleanValue(profileToUse.firstName),
                    last_name: cleanValue(profileToUse.lastName),
                    position: cleanValue(profileToUse.position),
                    headline: cleanValue(profileToUse.position),
                    company_name: cleanValue(profileToUse.company)
                },
                list_uuid: listUuid,
                update_if_exists: true,
                move_to_list: false
            };
            
            console.log('Lead data being sent to API:', JSON.stringify(leadData, null, 2));

            fetch('https://leadgen.grinfi.io/leads/api/leads/upsert', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${result.grinfiApiKey}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(leadData)
            })
            .then(response => {
                if (response.ok) {
                    return response.json();
                } else {
                    return response.text().then(text => {
                        throw new Error(`HTTP ${response.status}: ${text}`);
                    });
                }
            })
            .then(data => {
                showMessage(addMessage, 'Lead added successfully!', 'success');
            })
            .catch(error => {
                console.error('Error adding lead:', error);
                showMessage(addMessage, `Failed to add lead: ${error.message}`, 'error');
            })
            .finally(() => {
                resetAddButton();
            });
        });
    }

    function checkExistingLead(profile) {
        console.log('=== checkExistingLead called ===');
        console.log('Profile:', profile);
        
        chrome.storage.sync.get(['grinfiApiKey'], function(result) {
            if (!result.grinfiApiKey) {
                console.log('No API key found');
                return;
            }

            const lookupData = {
                linkedin_id: profile.linkedinUrl || window.location.href
            };

            console.log('Looking up lead with data:', lookupData);

            fetch('https://leadgen.grinfi.io/leads/api/leads/lookup-one', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${result.grinfiApiKey}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(lookupData)
            })
            .then(async response => {
                console.log('Lookup response status:', response.status);
                if (response.ok) {
                    return response.json();
                } else if (response.status === 422 || response.status === 404) {
                    // Lead not found - this means it's a NEW lead, so keep existingLead null
                    existingLead = null;
                    displayLeadNotFound();
                    // Re-display profile with edit button now visible
                    displayLinkedInProfile(currentProfile);
                    return null;
                } else {
                    const errorText = await response.text();
                    console.error('Lookup error:', response.status, errorText);
                    throw new Error(`HTTP ${response.status}: ${errorText}`);
                }
            })
            .then(data => {
                if (data) {
                    console.log('Raw lead data from API:', data);
                    existingLead = data;
                    displayExistingLead(data);
                    // Re-display profile without edit button for existing leads
                    displayLinkedInProfile(currentProfile);
                } else {
                    console.log('No lead data returned - this is a new lead');
                }
            })
            .catch(error => {
                console.error('Error checking existing lead:', error);
                existingLead = null;
                displayLeadNotFound();
                // Re-display profile with edit button for new leads
                displayLinkedInProfile(currentProfile);
            });
        });
    }

    function displayExistingLead(leadData) {
        console.log('=== LEAD DATA DEBUG ===');
        console.log('Raw lead data:', JSON.stringify(leadData, null, 2));
        console.log('Available lists:', JSON.stringify(lists, null, 2));
        console.log('Lead list_uuid:', leadData.lead?.list_uuid);
        console.log('Lead created_at:', leadData.lead?.created_at);
        
        // Try multiple ways to find the list name
        let listName = 'Unknown List';
        const listUuidToFind = leadData.lead?.list_uuid;
        
        // Wait for lists to load if they're not ready yet
        if ((!lists || lists.length === 0) && listUuidToFind) {
            console.log('Lists not loaded yet, using fallback list name');
            listName = `List ID: ${listUuidToFind}`;
        } else if (listUuidToFind && lists && lists.length > 0) {
            const foundList = lists.find(list => 
                (list.uuid && list.uuid === listUuidToFind) || 
                (list.id && list.id === listUuidToFind)
            );
            
            if (foundList) {
                listName = foundList.name || foundList.title || `List ${foundList.uuid || foundList.id}`;
                console.log('Found matching list:', foundList);
                
                // Pre-select this list in the dropdown
                listSelect.value = foundList.uuid || foundList.id;
            } else {
                console.log('Could not find list with UUID:', listUuidToFind);
                console.log('Available list UUIDs:', lists.map(l => l.uuid || l.id));
                listName = `List ID: ${listUuidToFind}`;
            }
        }

        // Get automation info
        let automationInfo = '';
        if (leadData.automations && leadData.automations.length > 0) {
            const automationNames = leadData.automations
                .map(auto => auto.name || auto.automation_name || 'Unknown Automation')
                .join(', ');
            automationInfo = `
                <div class="automation-info">
                    <strong>Active Automations:</strong> ${automationNames}
                </div>
            `;
        }

        // Format creation date
        let createdDate = 'Unknown';
        if (leadData.lead?.created_at) {
            try {
                const date = new Date(leadData.lead.created_at);
                createdDate = date.toLocaleDateString();
            } catch (e) {
                createdDate = leadData.lead.created_at;
            }
        }

        existingLeadInfo.innerHTML = `
            <div class="existing-lead found">
                <div class="lead-status found">✅ Profile found in Grinfi</div>
                <div class="lead-details">
                    <strong>Current list:</strong> <span class="lead-list-name">${listName}</span>
                </div>
                <div class="lead-details">
                    <strong>Added on:</strong> ${createdDate}
                </div>
                ${automationInfo}
                <div style="margin-top: 10px;">
                    <button id="view-in-grinfi-btn" style="
                        background-color: #0066cc;
                        color: white;
                        border: none;
                        padding: 8px 12px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 12px;
                        width: 100%;
                    ">🔗 View in Grinfi</button>
                </div>
            </div>
        `;
        
        existingLeadInfo.classList.remove('hidden');
        
        // Add event listener for "View in Grinfi" button
        setTimeout(() => {
            const viewBtn = document.getElementById('view-in-grinfi-btn');
            if (viewBtn && leadData.lead?.uuid) {
                viewBtn.addEventListener('click', function() {
                    const profileUrl = `https://leadgen.grinfi.io/leads/people/${leadData.lead.uuid}`;
                    window.open(profileUrl, '_blank');
                });
            }
        }, 100);
        
        // Change button to "Move to List" instead of "Add to List"
        addToListBtn.classList.add('hidden');
        moveToListBtn.classList.remove('hidden');
        moveToListBtn.disabled = false;
    }

    function displayLeadNotFound() {
        existingLeadInfo.innerHTML = `
            <div class="existing-lead not-found">
                <div class="lead-status not-found">➕ New profile - ready to add to Grinfi</div>
            </div>
        `;
        
        existingLeadInfo.classList.remove('hidden');
        
        // Show "Add to List" button
        addToListBtn.classList.remove('hidden');
        moveToListBtn.classList.add('hidden');
        addToListBtn.disabled = false;
    }

    function moveLeadToList(profile, listUuid) {
        chrome.storage.sync.get(['grinfiApiKey'], function(result) {
            if (!result.grinfiApiKey) {
                showMessage(addMessage, 'API key not found', 'error');
                resetMoveButton();
                return;
            }

            const leadData = {
                lead: {
                    linkedin_id: profile.linkedinUrl || window.location.href,
                    first_name: profile.firstName || '',
                    last_name: profile.lastName || '',
                    position: profile.position || '',
                    headline: profile.position || ''
                    // Note: Intentionally NOT including company_name for existing leads
                },
                list_uuid: listUuid,
                update_if_exists: true,
                move_to_list: true
            };
            
            console.log('Moving lead to new list:', JSON.stringify(leadData, null, 2));

            fetch('https://leadgen.grinfi.io/leads/api/leads/upsert', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${result.grinfiApiKey}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(leadData)
            })
            .then(response => {
                if (response.ok) {
                    return response.json();
                } else {
                    return response.text().then(text => {
                        throw new Error(`HTTP ${response.status}: ${text}`);
                    });
                }
            })
            .then(data => {
                showMessage(addMessage, 'Lead moved successfully!', 'success');
                // Refresh the lead info to show new list
                setTimeout(() => {
                    checkExistingLead(currentProfile);
                }, 1000);
            })
            .catch(error => {
                console.error('Error moving lead:', error);
                showMessage(addMessage, `Failed to move lead: ${error.message}`, 'error');
            })
            .finally(() => {
                resetMoveButton();
            });
        });
    }

    function resetAddButton() {
        addToListBtn.disabled = false;
        addToListBtn.textContent = 'Add to Selected List';
    }

    function resetMoveButton() {
        moveToListBtn.disabled = false;
        moveToListBtn.textContent = 'Move to Selected List';
    }

    function showMessage(element, message, type) {
        element.innerHTML = `<div class="${type}">${message}</div>`;
        setTimeout(() => {
            element.innerHTML = '';
        }, 5000);
    }

    function showStatus(message, type) {
        statusContainer.innerHTML = `<div class="status ${type}">${message}</div>`;
    }

    // Add refresh lists functionality
    document.getElementById('refresh-lists-btn').addEventListener('click', function() {
        console.log('Refreshing lists...');
        this.disabled = true;
        this.textContent = '⟳';
        
        loadLists()
            .then(() => {
                console.log('Lists refreshed successfully');
                showMessage(addMessage, 'Lists refreshed!', 'success');
            })
            .catch(error => {
                console.error('Error refreshing lists:', error);
                showMessage(addMessage, 'Failed to refresh lists', 'error');
            })
            .finally(() => {
                this.disabled = false;
                this.textContent = '🔄';
            });
    });

    function showFallbackProfileSection() {
        linkedinProfileInfo.innerHTML = `
            <div class="linkedin-info">
                <h4>LinkedIn Profile Detection</h4>
                <p>Unable to automatically extract profile data from this page.</p>
                <p>Make sure you're on a LinkedIn profile page (linkedin.com/in/username).</p>
                <p>You can try refreshing the page and reopening the extension.</p>
            </div>
        `;
        profileSection.classList.remove('hidden');
        addToListBtn.disabled = true;
        addToListBtn.textContent = 'Profile Data Required';
        
        // Still load lists for manual entry if needed
        loadLists();
    }
});