// Content script for LinkedIn profile pages
(function() {
    'use strict';

    // Function to extract profile data from LinkedIn page
    // Uses structural DOM traversal instead of CSS class selectors,
    // so it won't break when LinkedIn changes their obfuscated class names.
    function extractProfileData() {
        const profile = {};

        try {
            // Get current URL
            profile.linkedinUrl = window.location.href;

            // Extract LinkedIn username from URL
            const urlMatch = window.location.href.match(/linkedin\.com\/in\/([^\/\?]+)/);
            if (urlMatch) {
                profile.linkedinUsername = urlMatch[1];
            }

            // --- STRUCTURAL EXTRACTION ---
            // LinkedIn profile pages have a stable structure:
            // Inside <main>, the first <h2> is always the person's name.
            // Walking up from that h2, there's a container div with 3-4 children:
            //   children[0] (DIV) = Name + connection degree ("Name · 1st")
            //   children[1] (P)   = Headline/position
            //   children[2] (P)   = Company (+ optionally " · Education")
            //   children[3] (DIV) = Location + "Contact info" link

            const main = document.querySelector('main');
            const nameH2 = main ? main.querySelector('h2') : null;

            // Extract name from the first h2, fallback to page title
            let fullName = '';
            if (nameH2 && nameH2.textContent.trim()) {
                fullName = nameH2.textContent.trim();
            } else {
                // Fallback: page title is always "Name | LinkedIn"
                fullName = document.title.replace(/\s*\|.*$/, '').trim();
            }

            profile.name = fullName;

            // Split name into first and last
            if (fullName) {
                const cleanFullName = fullName.replace(/\s*[·•].*$/, '').replace(/\s*\d(st|nd|rd|th)$/i, '').trim();
                const nameParts = cleanFullName.split(' ');
                profile.firstName = nameParts[0] || '';
                profile.lastName = nameParts.slice(1).join(' ') || '';
            }

            // Find the profile info container by walking up from the name h2
            let profileContainer = null;
            if (nameH2) {
                let el = nameH2;
                for (let i = 0; i < 20; i++) {
                    el = el.parentElement;
                    if (!el) break;

                    const children = Array.from(el.children);
                    // The container has 3+ children with at least one <p> containing
                    // meaningful text (not just "· 1st" connection degree markers)
                    const hasMeaningfulP = children.some(c => {
                        if (c.tagName !== 'P') return false;
                        const t = c.textContent.trim();
                        return t.length > 10 && !/^[·\s\d]*(1st|2nd|3rd)/i.test(t);
                    });

                    if (children.length >= 3 && hasMeaningfulP) {
                        profileContainer = el;
                        break;
                    }
                }
            }

            if (profileContainer) {
                const children = Array.from(profileContainer.children);

                // children[1] = headline/position (P tag)
                if (children[1]) {
                    const headlineText = children[1].textContent.trim();
                    if (headlineText && !/^[·\s\d]*(1st|2nd|3rd)/i.test(headlineText)) {
                        profile.headline = headlineText;
                        profile.position = headlineText;
                    }
                }

                // children[2] = company name (P tag), may include " · Education"
                if (children[2]) {
                    const companyRaw = children[2].textContent.trim();
                    const company = companyRaw.split(/\s*·\s*/)[0].trim();
                    if (company && company.length > 1) {
                        profile.company = company;
                    }
                }

                // children[3] = location + "Contact info" (DIV)
                if (children[3]) {
                    const locationRaw = children[3].textContent.trim();
                    const location = locationRaw.replace(/·?\s*Contact info\s*$/i, '').trim();
                    if (location && location.length > 1) {
                        profile.location = location;
                    }
                }
            }

        } catch (error) {
            console.error('Error extracting profile data:', error);
        }

        // Debug information
        console.log('Grinfi Extension: Extracted profile data:', profile);
        console.log('Grinfi Extension: Current URL:', window.location.href);
        console.log('Grinfi Extension: Is profile page:', isLinkedInProfilePage());

        return profile;
    }
    
    // Function to check if we're on a LinkedIn profile page
    function isLinkedInProfilePage() {
        return (window.location.href.includes('linkedin.com/in/') || 
                window.location.pathname.startsWith('/in/')) && 
               !window.location.href.includes('/edit/');
    }
    
    // Listen for messages from popup
    chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
        if (request.action === 'getProfileData') {
            if (isLinkedInProfilePage()) {
                const profileData = extractProfileData();
                sendResponse({profile: profileData});
            } else {
                sendResponse({error: 'Not on a LinkedIn profile page'});
            }
        }
        return true; // Keep message channel open for async response
    });
    
    // Optional: Add a visual indicator when the extension is active
    function addVisualIndicator() {
        if (!isLinkedInProfilePage()) return;
        
        // Check if indicator already exists
        if (document.getElementById('grinfi-indicator')) return;
        
        const indicator = document.createElement('div');
        indicator.id = 'grinfi-indicator';
        indicator.style.cssText = `
            position: fixed;
            top: 10px;
            right: 10px;
            background: #0066cc;
            color: white;
            padding: 8px 12px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
            z-index: 9999;
            box-shadow: 0 2px 8px rgba(0,0,0,0.2);
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        `;
        indicator.textContent = 'Grinfi Extension Active';
        
        document.body.appendChild(indicator);
        
        // Remove indicator after 3 seconds
        setTimeout(() => {
            if (indicator.parentNode) {
                indicator.parentNode.removeChild(indicator);
            }
        }, 3000);
    }
    
    // Initialize when page loads
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', addVisualIndicator);
    } else {
        addVisualIndicator();
    }
    
    // Re-check when navigating within LinkedIn (SPA behavior)
    let currentUrl = window.location.href;
    new MutationObserver(() => {
        if (window.location.href !== currentUrl) {
            currentUrl = window.location.href;
            setTimeout(addVisualIndicator, 1000); // Wait for page to load
        }
    }).observe(document, {subtree: true, childList: true});
    
})();